document.addEventListener("DOMContentLoaded", function () {
    // Mobile menu toggle
    const menuToggle = document.querySelector(".menu-toggle");
    const navLinks = document.querySelector(".nav-links");

    if (menuToggle && navLinks) {
        menuToggle.addEventListener("click", function () {
            navLinks.classList.toggle("active");
            const icon = menuToggle.querySelector("i");
            if (navLinks.classList.contains("active")) {
                icon.classList.remove("fa-bars");
                icon.classList.add("fa-times");
            } else {
                icon.classList.remove("fa-times");
                icon.classList.add("fa-bars");
            }
        });
    }

    // Close mobile menu when clicking on a link
    const navLinksItems = document.querySelectorAll(".nav-links a");
    navLinksItems.forEach(link => {
        link.addEventListener("click", () => {
            if (navLinks.classList.contains("active")) {
                navLinks.classList.remove("active");
                const icon = menuToggle.querySelector("i");
                icon.classList.remove("fa-times");
                icon.classList.add("fa-bars");
            }
        });
    });

    // Policy search form validation and submission
    const searchForm = document.querySelector(".search-form");
    const mobileInput = document.querySelector("#mobile");
    const errorMessage = document.querySelector("#mobile-error");

    if (searchForm && mobileInput && errorMessage) {
        searchForm.addEventListener("submit", function (event) {
            event.preventDefault();

            const mobileValue = mobileInput.value.trim();

            // Reset error message
            errorMessage.textContent = "";
            errorMessage.style.display = "none";

            // Validate mobile number
            if (!/^\d{10}$/.test(mobileValue)) {
                errorMessage.textContent = "Please enter a valid 10-digit mobile number.";
                errorMessage.style.display = "block";
                mobileInput.focus();
                return;
            }

            // Show loading state
            const submitButton = searchForm.querySelector('button[type="submit"]');
            const originalText = submitButton.innerHTML;
            submitButton.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Searching...';
            submitButton.disabled = true;

            // Fetch user data
            fetch(`admin-panel/fetchRecords.php?mobile=${mobileValue}`)
                .then((response) => response.json())
                .then((data) => {
                    if (data.error) {
                        errorMessage.textContent = "No policy found for this mobile number. Please check and try again.";
                        errorMessage.style.display = "block";
                    } else {
                        // Store the mobile number in local storage
                        localStorage.setItem("searchedMobileNumber", mobileValue);
                        // Redirect to the active plan page
                        window.location.href = "active-plan/index.html";
                    }
                })
                .catch((error) => {
                    console.error("Error fetching user data:", error);
                    errorMessage.textContent = "An error occurred. Please try again later.";
                    errorMessage.style.display = "block";
                })
                .finally(() => {
                    // Reset button state
                    submitButton.innerHTML = originalText;
                    submitButton.disabled = false;
                });
        });

        // Real-time mobile number validation
        mobileInput.addEventListener("input", function () {
            const value = this.value.replace(/\D/g, ''); // Remove non-digits
            this.value = value.slice(0, 10); // Limit to 10 digits
            
            if (errorMessage.style.display === "block") {
                errorMessage.style.display = "none";
            }
        });
    }

    // Enhanced Counter Animation for Growing Numbers
    const counters = document.querySelectorAll(".counter");
    const animatedCounters = new Set();

    function animateCounter(counter) {
        if (animatedCounters.has(counter)) return;
        animatedCounters.add(counter);

        const target = parseInt(counter.getAttribute("data-target"));
        const duration = 2000; // 2 seconds
        const startTime = performance.now();
        
        function updateCounter(currentTime) {
            const elapsed = currentTime - startTime;
            const progress = Math.min(elapsed / duration, 1);
            
            // Easing function for smooth animation
            const easeOutQuart = 1 - Math.pow(1 - progress, 4);
            const currentValue = Math.round(target * easeOutQuart);
            
            counter.textContent = currentValue.toLocaleString();
            
            if (progress < 1) {
                requestAnimationFrame(updateCounter);
            } else {
                counter.textContent = target.toLocaleString();
            }
        }
        
        requestAnimationFrame(updateCounter);
    }

    // Intersection Observer for counter animation
    const counterObserver = new IntersectionObserver((entries) => {
        entries.forEach(entry => {
            if (entry.isIntersecting) {
                animateCounter(entry.target);
            }
        });
    }, { threshold: 0.5 });

    counters.forEach(counter => {
        counter.textContent = "0";
        counterObserver.observe(counter);
    });

    // FAQ Toggle functionality
    const faqQuestions = document.querySelectorAll(".faq-question");

    faqQuestions.forEach((question) => {
        question.addEventListener("click", function () {
            const answer = this.nextElementSibling;
            const icon = this.querySelector("i");
            const isOpen = answer.style.display === "block";

            // Close all other FAQ items
            faqQuestions.forEach((otherQuestion) => {
                if (otherQuestion !== question) {
                    const otherAnswer = otherQuestion.nextElementSibling;
                    const otherIcon = otherQuestion.querySelector("i");
                    otherAnswer.style.display = "none";
                    otherIcon.classList.remove("fa-chevron-up");
                    otherIcon.classList.add("fa-chevron-down");
                }
            });

            // Toggle current FAQ item
            if (isOpen) {
                answer.style.display = "none";
                icon.classList.remove("fa-chevron-up");
                icon.classList.add("fa-chevron-down");
            } else {
                answer.style.display = "block";
                icon.classList.remove("fa-chevron-down");
                icon.classList.add("fa-chevron-up");
            }
        });
    });

    // Quote form handling
    const quoteForm = document.querySelector(".quote-form");
    if (quoteForm) {
        quoteForm.addEventListener("submit", function (event) {
            event.preventDefault();
            
            const formData = new FormData(quoteForm);
            const submitButton = quoteForm.querySelector('button[type="submit"]');
            const originalText = submitButton.innerHTML;
            
            // Show loading state
            submitButton.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Submitting...';
            submitButton.disabled = true;
            
            // Simulate form submission (replace with actual API call)
            setTimeout(() => {
                alert("Thank you for your interest! Our team will contact you within 24 hours.");
                quoteForm.reset();
                submitButton.innerHTML = originalText;
                submitButton.disabled = false;
            }, 2000);
        });
    }

    // Contact form handling
    const contactForm = document.querySelector(".contact-form form");
    if (contactForm) {
        contactForm.addEventListener("submit", function (event) {
            event.preventDefault();
            
            const submitButton = contactForm.querySelector('button[type="submit"]');
            const originalText = submitButton.innerHTML;
            
            // Show loading state
            submitButton.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Sending...';
            submitButton.disabled = true;
            
            // Simulate form submission (replace with actual API call)
            setTimeout(() => {
                alert("Thank you for your message! We'll get back to you soon.");
                contactForm.reset();
                submitButton.innerHTML = originalText;
                submitButton.disabled = false;
            }, 2000);
        });
    }

    // Back to top button
    const backToTopButton = document.getElementById("back-to-top");
    if (backToTopButton) {
        window.addEventListener("scroll", function () {
            if (window.pageYOffset > 300) {
                backToTopButton.classList.add("show");
            } else {
                backToTopButton.classList.remove("show");
            }
        });

        backToTopButton.addEventListener("click", function () {
            window.scrollTo({
                top: 0,
                behavior: "smooth"
            });
        });
    }

    // Smooth scrolling for anchor links
    const anchorLinks = document.querySelectorAll('a[href^="#"]');
    anchorLinks.forEach(link => {
        link.addEventListener("click", function (event) {
            const targetId = this.getAttribute("href");
            if (targetId === "#") return;
            
            const targetElement = document.querySelector(targetId);
            if (targetElement) {
                event.preventDefault();
                const offsetTop = targetElement.offsetTop - 80; // Account for fixed navbar
                
                window.scrollTo({
                    top: offsetTop,
                    behavior: "smooth"
                });
            }
        });
    });

    // Add loading states to all buttons
    const buttons = document.querySelectorAll(".btn");
    buttons.forEach(button => {
        if (button.getAttribute("href") && !button.getAttribute("href").startsWith("#")) {
            button.addEventListener("click", function () {
                const originalText = this.innerHTML;
                this.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Loading...';
                this.style.pointerEvents = "none";
                
                // Reset after 3 seconds if navigation doesn't occur
                setTimeout(() => {
                    this.innerHTML = originalText;
                    this.style.pointerEvents = "auto";
                }, 3000);
            });
        }
    });

    // Navbar background change on scroll
    const navbar = document.querySelector(".navbar");
    if (navbar) {
        window.addEventListener("scroll", function () {
            if (window.pageYOffset > 100) {
                navbar.style.background = "rgba(255, 255, 255, 0.95)";
                navbar.style.backdropFilter = "blur(10px)";
            } else {
                navbar.style.background = "#ffffff";
                navbar.style.backdropFilter = "none";
            }
        });
    }

    // Animate elements on scroll
    const animateElements = document.querySelectorAll(".insurance-plan-card, .service-card, .benefit-card, .testimonial-card");
    
    const animateObserver = new IntersectionObserver((entries) => {
        entries.forEach((entry, index) => {
            if (entry.isIntersecting) {
                setTimeout(() => {
                    entry.target.style.opacity = "1";
                    entry.target.style.transform = "translateY(0)";
                }, index * 100); // Stagger animation
            }
        });
    }, { threshold: 0.1 });

    animateElements.forEach(element => {
        element.style.opacity = "0";
        element.style.transform = "translateY(20px)";
        element.style.transition = "opacity 0.6s ease, transform 0.6s ease";
        animateObserver.observe(element);
    });

    // Form input enhancements
    const formInputs = document.querySelectorAll('input, textarea, select');
    formInputs.forEach(input => {
        input.addEventListener('focus', function () {
            this.parentElement.classList.add('focused');
        });
        
        input.addEventListener('blur', function () {
            if (!this.value) {
                this.parentElement.classList.remove('focused');
            }
        });
        
        // Check for pre-filled values
        if (input.value) {
            input.parentElement.classList.add('focused');
        }
    });

    // Phone number formatting
    const phoneInputs = document.querySelectorAll('input[type="tel"]');
    phoneInputs.forEach(input => {
        input.addEventListener('input', function () {
            let value = this.value.replace(/\D/g, '');
            if (value.length > 10) value = value.slice(0, 10);
            this.value = value;
        });
    });

    // Auto-hide alerts/notifications
    const alerts = document.querySelectorAll('.alert, .notification');
    alerts.forEach(alert => {
        setTimeout(() => {
            alert.style.opacity = '0';
            setTimeout(() => {
                alert.remove();
            }, 300);
        }, 5000);
    });
});
